function [CiMax Qmax] = modularity_dir_SA(A)
%MODULARITY_DIR_SA Find partition which maximizes modularity using
%simulated annealing algorithm
%   Ci is a vector which shows the module membership of each row of A
%   Q is the modularity of this partition
%
%   Note: A should be a square matrix
%
%   This method implements the algorithm of Guimera and Amaral in
%   'Functional cartography of complex metabolic networks', Nature, Feb.
%   2005

N = size(A,1);

% Start out with a random partitioning of network with n groups
n = randi(N,1);
Ci = randi(n,[N 1]);
CiTest = Ci;


% Initialize temperature - these can be futzed around with
T = .5;
f = 1;
relax = .999;
% Initialize number of steps for decreasing the temperature
nSteps = 5000;

nNodeMvts = f*N^2;
nCollectiveMvts = f*N;

kout = sum(A,2);
kin = sum(A,1);
m = sum(sum(A));
kMat = 1/m .* kout * kin;
currQ = Q_dir(Ci);
Qmax = currQ;
CiMax = Ci;

for i = 1:nSteps
    % fprintf('\n\nModule assignments: ');
    % fprintf('%u ',Ci);
    % fprintf('\n');
    % fprintf('Current modularity: %f\n',currQ);
    % fprintf('Current temperature: %f\n',T);
    % Do the individual node movements
    
    
    for j = 1:nNodeMvts
        
        
        if max(Ci)>1
            
            idx = randi(N,1);
            % Randomly assign this node to a different group
            newMod = randsample(setdiff(1:max(Ci),Ci(idx)),1);
            CiTest = Ci;
            CiTest(idx) = newMod;
            %         fprintf('Reassigning node %u to module %u\n',idx, newMod);
            % Check for empty modules, and compress as needed
            fixEmptyModules;
            % Calculate new modularity
            testQ = Q_dir(CiTest);
            % Decide whether to accept or reject this move
            acceptOrReject;
        end
        if currQ>Qmax
            Qmax = currQ;
            CiMax = Ci;
        end
    end
    
    
    for j = 1:nCollectiveMvts
        
        % Decide if we're doing a split or a merge, with prob 1/2
        if rand()<.5  %Do a split
            while true  % Loop until we get a module with more than 1 element
                mod = randi(max(Ci),1);
                if sum(Ci==mod) > 1
                    break;
                end
            end
            
            CiTest = Ci;
            % Pick the nodes to split off
            toChange = ~(double(CiTest == mod) - (randi(2,[size(CiTest),1])-1)) & (CiTest == mod);
            CiTest(toChange) = max(CiTest) + 1;
            %             fprintf('Splitting module %u\n',mod);
            % Check for empty modules, and compress as needed
            fixEmptyModules;
            % Calculate new modularity
            testQ = Q_dir(CiTest);
            % Decide whether to accept or reject this move
            acceptOrReject;
            
        else % do a merge
            if max(Ci)>1
                mods = randsample(1:max(Ci),2);
                CiTest = Ci;
                CiTest(CiTest==max(mods)) = min(mods);  % Join modules
                %                 fprintf('Joining modules %u and %u\n',mods(1),mods(2));
                % Check for empty modules, and compress as needed
                fixEmptyModules;
                
                % Calculate new modularity
                testQ = Q_dir(CiTest);
                
                % Decide whether to accept or reject this move
                acceptOrReject;
                
            end
        end
        if currQ>Qmax
            Qmax = currQ;
            CiMax = Ci;
        end
    end
    T = relax*T;
end

Q = Q_dir(Ci);

    function Qt = Q_dir(CtoUse)
        
        s = repmat(CtoUse,[1 size(CtoUse)]);
        D = ~(s-s');
        B = A - kMat;
        B = B+B';
        Qt = sum(sum(B.*D))/(2*m);
    end

    function fixEmptyModules
        % Note: this function will operate on CiTest
        while true
            e = setdiff(1:max(CiTest),CiTest);
            if ~isempty(e)
                CiTest(CiTest==max(CiTest)) = min(e);
            else
                return
            end
        end
    end

    function acceptOrReject
        %         fprintf('CurrQ: %f\tTestQ: %f\n',currQ,testQ);
        if (testQ>currQ)
            Ci = CiTest;
            currQ = testQ;
            %             disp('Accepted');
        else
            if rand() < exp(-(currQ-testQ)/T)
                Ci = CiTest;
                currQ = testQ;
                %                 disp('Accepted despite bad modularity');
            else
                %                 disp('Rejected');
            end
            
        end
    end

end